from os import linesep
from sys import argv, stdin


def converter(direction, filename):
    """
    The function prints the content of the file converted either to upper or lower case.
    :param direction: The conversion direction; 'U' for uppercase, 'L' for lowercase
    :param filename: The name of the file to be converted and printed
    :return: nothing
    """
    with open(filename, 'rt') as f:
        lines = f.readlines()
        if direction == 'U':
            for line in lines:
                print(line.upper(), end='')
        else:
            for line in lines:
                print(line.lower(), end='')


def cut(args):
    """
    Simplified version of the unix cut command.
    It reads data from the stdin and prints out the result to stdout. It expects two parameters in
    the given order: cut -d delim -f fields. Technically there are four parameters since the name and
    the data are also delimited by a space.
    Note that the fields are indexed starting at 1, contrary to zero-based-indexed lists in Python.
    :param args: the list of command-line arguments
    :return: nothing
    """
    delim = args[2]  # the delimiter
    ranges = args[4].split(',')  # list of ranges, each one can be a single field or a range M-N (-N, N-)

    fields = [r.split('-') for r in ranges]

    for line in stdin.readlines():
        line_fields = line.strip(linesep).split(delim)  # the new line character is removed

        for rng in fields:
            try:
                if len(rng) == 1:  # a single field, not a range
                    try:
                        print(line_fields[int(rng[0]) - 1], end=' ')
                    except IndexError:  # if the line is shorter, do not print anything out
                        pass
                else:  # a range in the form M-N
                    for i in range(0 if rng[0] == '' else int(rng[0]) - 1,
                                   len(line_fields) if rng[1] == '' else int(rng[1])):
                        try:
                            print(line_fields[i], end=' ')
                        except IndexError:  # if the line is shorter, do not print anything out
                            pass

            except ValueError:  # the index is not an integer, terminating
                print()
                print('Invalid field value "{}"'.format(rng[0]))
                return

        print()


def todo(args):
    """
    A simple scheduler (a "TODO" list).
    The data are stored in a file and manipulated via parameters of the command line.
    :param args: the list of command-line arguments
    :return: nothing    
    """
    FILENAME = 'todo.lst'
    records = list()

    def read_data():
        nonlocal FILENAME
        nonlocal records
        with open(FILENAME, 'rt') as f:
            lines = f.readlines()
            #            This is an alternative for the single complex line below.
            #            Be careful with complex one-line statements, they tend to be hard to read.
            #            for line in lines:
            #                pos = line.find(':')
            #                records.append((int(line[0:pos]), line[pos + 1:].strip(linesep)))
            records = [(int(item[0:item.find(':')]), item[item.find(':') + 1:].strip(linesep)) for item in lines]

    def store_data():
        nonlocal FILENAME
        nonlocal records
        with open(FILENAME, 'wt') as f:
            for record in records:
                print('{}:{}'.format(record[0], record[1]), file=f)

    def add_record(priority, text):
        nonlocal records
        records.append((priority, text))

    def list_ascending():
        nonlocal records
        records.sort(key=lambda x: x[0])  # the lambda function selects the first item of the tuple, i.e., the priority
        # alternatively, you can define your own function that accepts a tuple and returns its first item

        for record in records:
            print("{}: {}".format(record[0], record[1]))

    def list_descending():
        nonlocal records
        records.sort(key=lambda x: x[0], reverse=True)

        for record in records:
            print('{}: {}'.format(record[0], record[1]))

    def delete_record():
        nonlocal records

        for i in range(len(records)):
            print('{:2d}> {}: {}'.format(i, records[i][0], records[i][1]))

        while True:
            choice = input('Enter record id to be deleted: ')
            try:
                rec_id = int(choice)
                if 0 <= rec_id < len(records):
                    break
                else:
                    raise ValueError

            except ValueError:
                print('You have to enter id between 0 and {}.'.format(len(records) - 1))

        del records[rec_id]
        print('Record deleted.')

    read_data()  # load the data from file

    if len(args) < 2:
        print('Not enough arguments, exiting.')
        return

    if args[1] == '-a':  # Add a new record
        if len(args) < 4:
            print('Not enough arguments, exiting.')
            return

        try:
            add_record(int(args[2]), args[3])
            print('Record added.')
        except ValueError:
            print('Priority has to be a number.')
            return

    elif args[1] == '-l':  # print descending
        list_descending()
        return

    elif args[1] == '-r':  # print ascending
        list_ascending()
        return

    elif args[1] == '-d':  # delete a record
        delete_record()

    else:
        print('Invalid arguments provided.')
        return

    store_data()  # store the result to a file if needed


# converter
# if (len(argv) == 3) and (argv[1] == '-L' or argv[1] == '-U'):
#   converter(argv[1][1], argv[2])

# cut
# if len(argv) == 5:
#    cut(argv)

# todolist
todo(argv)
