﻿using BenchmarkDotNet.Attributes;
using BenchmarkDotNet.Running;
using System.Runtime.CompilerServices;

BenchmarkRunner.Run<Benchmarks>();

public class Benchmarks {
	private string _tempFilePath = null!;
	private readonly int[] _sink = new int[256];

	[GlobalSetup]
	public void Setup() {
		// Create a temporary file
		_tempFilePath = Path.GetTempFileName();
		File.WriteAllBytes(_tempFilePath, Enumerable.Range(0, 64 * 1024 * 1024).Select(x => (byte) x).ToArray());
	}

	[GlobalCleanup]
	public void Cleanup() {
		if (File.Exists(_tempFilePath)) {
			File.Delete(_tempFilePath);
		}
	}

	[Benchmark]
	public void FileStream_ReadPerOneByte() {
		using var fs = new FileStream(_tempFilePath, FileMode.Open, FileAccess.Read);
		int byteValue;
		while ((byteValue = fs.ReadByte()) >= 0) {
			_sink[byteValue]++;
		}
	}

	[Benchmark]
	public void FileStream_ReadPer4KBuffer_ByteArray() {
		using var fs = new FileStream(_tempFilePath, FileMode.Open, FileAccess.Read);
		var buffer = new byte[4096];
		int bytesRead;
		while ((bytesRead = fs.Read(buffer, 0, buffer.Length)) > 0) {
			for (int i = 0; i < bytesRead; ++i) {
				_sink[buffer[i]]++;
			}
		}
	}

	[Benchmark]
	public void FileStream_ReadPer4KBuffer_ByteSpan() {
		using var fs = new FileStream(_tempFilePath, FileMode.Open, FileAccess.Read);
		var buffer = new byte[4096];
		int bytesRead;
		while ((bytesRead = fs.Read(buffer)) > 0) {
			for (int i = 0; i < bytesRead; ++i) {
				_sink[buffer[i]]++;
			}
		}
	}

	[Benchmark]
	public void FileStream_ReadPer4KBuffer_CustomGetByte() {
		using var fs = new FileStream(_tempFilePath, FileMode.Open, FileAccess.Read);
		var buffer = new byte[4096];
		int bytesRead;
		while ((bytesRead = fs.Read(buffer, 0, buffer.Length)) > 0) {
			for (int i = 0; i < bytesRead; ++i) {
				_sink[GetByte(buffer, i)]++;
			}
		}
	}

	[MethodImpl(MethodImplOptions.NoInlining)]
	private byte GetByte(byte[] buffer, int offset) {
		return buffer[offset];
	}

	[Benchmark]
	public void BinaryReader_ReadPerOneByte() {
		using var fs = new FileStream(_tempFilePath, FileMode.Open, FileAccess.Read);
		using var br = new BinaryReader(fs);

		try {
			while (true) {
				_sink[br.ReadByte()]++;
			}
		} catch (EndOfStreamException) {
		}
	}

	[Benchmark]
	public void FileStream_ReadPer1KBuffer_ByteArray() {
		using var fs = new FileStream(_tempFilePath, FileMode.Open, FileAccess.Read);
		var buffer = new byte[1024];
		int bytesRead;
		while ((bytesRead = fs.Read(buffer, 0, buffer.Length)) > 0) {
			for (int i = 0; i < bytesRead; ++i) {
				_sink[buffer[i]]++;
			}
		}
	}

	[Benchmark]
	public void FileStream_ReadPer1000ByteBuffer_ByteArray() {
		using var fs = new FileStream(_tempFilePath, FileMode.Open, FileAccess.Read);
		var buffer = new byte[1000];
		int bytesRead;
		while ((bytesRead = fs.Read(buffer, 0, buffer.Length)) > 0) {
			for (int i = 0; i < bytesRead; ++i) {
				_sink[buffer[i]]++;
			}
		}
	}
}
