﻿namespace Huffman.Common;

public class HuffmanProgram
{
    private readonly string[] _args;
    public HuffmanProgram(string[] args, int expectedArgs)
    {
        if (args.Length != expectedArgs) {
            HuffmanProgramException.ThrowArgumentsLengthMismatch();
        }

        _args = args;
    }

    public Stream OpenInputStream(int index) => TryOpenFileStream(_args[index], FileMode.Open, FileAccess.Read);
    // public Stream OpenOutputStream(int index) => TryOpenFileStream(_args[index], FileMode.CreateNew, FileAccess.Write);

    private static FileStream TryOpenFileStream(string filePath, FileMode mode, FileAccess access)
    {
        try {
            return new FileStream(filePath, mode, access);
        } catch (Exception ex) {
            if (ex is FileNotFoundException or IOException or UnauthorizedAccessException or System.Security.SecurityException) {
                HuffmanProgramException.ThrowIOError();
            }
            throw;
        }
    }

    public long[] CountSymbolsInStream(Stream stream) => CountSymbolsInStreamV1(stream);

    private static long[] CountSymbolsInStreamV1(Stream stream)
    {
        long[] counts = new long[256];

        int symbol;
        while ((symbol = stream.ReadByte()) != -1) {
            counts[symbol]++;
        }

        return counts;
    }

    private static long[] CountSymbolsInStreamV2(Stream stream)
    {
        long[] counts = new long[256];

        var buffer = new byte[4096];
        int bytesRead;
        while ((bytesRead = stream.Read(buffer)) > 0) {
            for (int i = 0; i < bytesRead; ++i) {
                counts[buffer[i]]++;
            }
        }

        return counts;
    }
}

public class HuffmanProgramException(string message) : Exception(message)
{
    public static void ThrowArgumentsLengthMismatch() => throw new HuffmanProgramException("Argument Error");
    public static void ThrowIOError() => throw new HuffmanProgramException("File Error");
}
