﻿namespace Huffman.Common;

public class HuffmanTree {
    public Node RootNode { get; }

    private HuffmanTree(Node rootNode) {
        RootNode = rootNode;
    }

    public static HuffmanTree CreateFromSymbolCounts(long[] symbolCounts) {
        if (symbolCounts.Length != 256) {
            throw new ArgumentException("Count for all symbols (all 256 possible byte values) must be always provided.", nameof(symbolCounts));
        }

        var treeBuilder = new TreeBuilder(symbolCounts);
        var tree = new HuffmanTree(treeBuilder.BuildTree());

        return tree;
    }

    private class TreeBuilder(long[] symbolCounts) {
        private readonly Queue<Node> _leafQueue = new(PrepareLeafNodes(symbolCounts));
        private readonly Queue<Node> _innerNodeQueue = new();

        private static List<Node> PrepareLeafNodes(long[] symbolCounts)
        {
            var leafNodes = new List<Node>();

            for (int i = 0; i < symbolCounts.Length; i++) {
                if (symbolCounts[i] > 0) {
                    leafNodes.Add(Node.CreateLeaf(symbolCounts[i], (byte) i));
                }
            }

            leafNodes.Sort();
            return leafNodes;
        }

        /// <summary>
        /// Generate new Huffman tree based on symbol counts passed to constructor.
        /// </summary>
        /// <returns>Root node of the newly created Huffman tree.</returns>
        public Node BuildTree() {
            while (_leafQueue.Count > 0 || _innerNodeQueue.Count > 1) {
                var min1 = GetNextMinimalNode();
                var min2 = GetNextMinimalNode();

                var node = Node.CreateIntermediate(
                    weight: min1.Weight + min2.Weight,
                    left: min1,
                    right: min2
                );

                _innerNodeQueue.Enqueue(node);
            }

            return _innerNodeQueue.Dequeue();
        }

        private Node GetNextMinimalNode() {
            if (_innerNodeQueue.Count == 0 || (_leafQueue.Count > 0 && _leafQueue.Peek().Weight <= _innerNodeQueue.Peek().Weight)) {
                return _leafQueue.Dequeue();
            }

            return _innerNodeQueue.Dequeue();
        }
    }
}
