﻿namespace Huffman.Common;

public class Node : IComparable<Node> {
    /// <summary>
    /// Reference to left child node. Is always <code>null</code> for leaf nodes.
    /// </summary>
    public Node? Left { get; }

    /// <summary>
    /// Reference to right child node. Is always <code>null</code> for leaf nodes.
    /// </summary>
    public Node? Right { get; }

    /// <summary>
    /// Weight of the node:
    /// - For leaf nodes => frequency of the <see cref="Symbol"/>.
    /// - For inner nodes => <code>Left.Weight + Right.Weight</code>
    /// </summary>
    public long Weight { get; }

    /// <summary>
    /// A byte value this node represents. This property has a valid value only for leaf nodes and has no meaning in intermediate nodes.
    /// </summary>
    public byte Symbol { get; }

    private Node(Node? left, Node? right, long weight, byte symbol) {
        Left = left;
        Right = right;
        Weight = weight;
        Symbol = symbol;
    }

    public static Node CreateIntermediate(long weight, Node left, Node right) {
        ArgumentNullException.ThrowIfNull(left);
        ArgumentNullException.ThrowIfNull(right);
        ArgumentOutOfRangeException.ThrowIfLessThanOrEqual(weight, 0);

        return new Node(left, right, weight, 0 /* unused */);
    }

    public static Node CreateLeaf(long weight, byte symbol) {
        ArgumentOutOfRangeException.ThrowIfLessThanOrEqual(weight, 0);

        return new Node(null, null, weight, symbol);
    }

    /// <summary>
    /// Compare leaf nodes according to Huffman rules. Warning: This method defines a valid relationship only between two leaf nodes.
    /// </summary>
    /// <exception cref="System.InvalidOperationException">Thrown when <code>this</code> or <code>node</code> argument represent intermediate nodes.</exception>
    /// <exception cref="ArgumentNullException">Thrown when <paramref name="node"/> is null.</exception>
    public int CompareTo(Node? node) {
        ArgumentNullException.ThrowIfNull(node);

        if (Left is not null || node.Left is not null /* || Right is not null || node?.Right is not null */) {
            // No need to check for Right, due to how Huffman nodes work.
            throw new InvalidOperationException("CompareTo method can be only called on and only with leaf nodes.");
        }

        return Weight == node.Weight
            ? Symbol.CompareTo(node.Symbol)
            : Weight.CompareTo(node.Weight);
    }
}