﻿using Huffman.Common;
namespace Huffman.DebugPrintApp;

internal static class Program {
    public static void Main(string[] args) {
        try {
            // Step 1: parse input arguments and validate
            var huffmanProgram = new HuffmanProgram(args, 1);

            // Step 2: open input (and output in future) files & detect errors (thrown as custom HuffmanProgramException)
            using var inputStream = huffmanProgram.OpenInputStream(0);

			// Step 3: compression
			{
				// Substep 3.1: prepare huffman encoding tree
				// Substep 3.1.1:
				var symbolCounts = huffmanProgram.CountSymbolsInStream(inputStream);
				// Substep 3.1.2:
				var huffmanTree = HuffmanTree.CreateFromSymbolCounts(symbolCounts);

				// Substep 3.1.3: DebugPrintApp for Huffman I solution only:
				PrintInPreorder_V1(huffmanTree, Console.Out);
			}
        } catch (HuffmanProgramException ex) {
            Console.WriteLine(ex.Message);
        }
    }

    private static void PrintInPreorder_V1(HuffmanTree tree, TextWriter writer) {	// V1 solution = removes warning produced by version V0 below
        Stack<Node> stack = [];
        stack.Push(tree.RootNode);

        while (stack.Count > 0) {
            var currentNode = stack.Pop();

            if (currentNode.Left is null) {	// For leaf nodes both .Left and .Right all null - it is sufficient to test only one. 
			    // Leaf node
				writer.Write($"*{currentNode.Symbol}:{currentNode.Weight}");
            } else {
                // Inner node
                stack.Push(currentNode.Right!); // If .Left is not null, then .Right is not null either - see note below.
                stack.Push(currentNode.Left);
                writer.Write(currentNode.Weight);
            }

            if (stack.Count > 0) {
                writer.Write(' ');
            }
        }
    }

	private static void PrintInPreorder_V0(HuffmanTree tree, TextWriter writer) {
		Stack<Node> stack = [];
		stack.Push(tree.RootNode);

		while (stack.Count > 0) {
			var currentNode = stack.Pop();

			if (currentNode.Left is null) { 
                // Leaf node
				writer.Write($"*{currentNode.Symbol}:{currentNode.Weight}");
			} else {
                // Inner node
				stack.Push(currentNode.Right);  // Compiler warning: Possible null reference argument for parameter 'item' in 'void Stack<Node>.Push(Node item)'.
												// But we know, this is not possible: if currentNode.Left is not null, we have inner node, and then currentNode.Right is not null either!
                                                // There are 2 solutions, how to tell this information to the compiler - V1 above, or V2 below
				stack.Push(currentNode.Left);
				writer.Write(currentNode.Weight);
			}

			if (stack.Count > 0) {
				writer.Write(' ');
			}
		}
	}

	private static void PrintInPreorder_V2(HuffmanTree tree, TextWriter writer) {   // V2 solution = removes warning produced by version V0 above
		Stack<Node> stack = [];
		stack.Push(tree.RootNode);

		while (stack.Count > 0) {
			var currentNode = stack.Pop();

			if (currentNode.Left is null || currentNode.Right is null) {    // currentNode.Right being null is implied but the compiler wouldn't know at (*)
				// Leaf node
				writer.Write($"*{currentNode.Symbol}:{currentNode.Weight}");
			} else {
				// Inner node
				stack.Push(currentNode.Right); // (*)
				stack.Push(currentNode.Left);
				writer.Write(currentNode.Weight);
			}

			if (stack.Count > 0) {
				writer.Write(' ');
			}
		}
	}
}
