using ExpressionEvaluatorI_2024;
using Xunit;

namespace ExpressionEvaluatorI_2024_UnitTests;

public class ExpressionParserTests {

    // Parsing all value types

    [Fact]
    [Trait("Category", "Basic-Values")]
    public void ParsePrefixExpression_ConstantValue_OneDigit() {
        // Arrange
        var prefixFormula = "1";
        var expectedExpr = new ConstantExpression(1);

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Basic-Values")]
    public void ParsePrefixExpression_ConstantValue_ThreeDigits() {
        // Arrange
        var prefixFormula = "123";
        var expectedExpr = new ConstantExpression(123);

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Basic-Values")]
    public void ParsePrefixExpression_ConstantValue_MaximumSupportedValue() {
        // Arrange
        const int maximumSupportedValue = int.MaxValue;

        var prefixFormula = maximumSupportedValue.ToString();
        var expectedExpr = new ConstantExpression(maximumSupportedValue);

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_TooBigConstant_ShouldReturnNull() {
        // Arrange
        var prefixFormula = 4_000_000_000.ToString();

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_NegativeConstant_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "-123";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    // Parsing all binary operators

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void ParsePrefixExpression_SinglePlusOperator_ConstantOperands() {
        // Arrange
        var prefixFormula = "+ 1 2";
        var expectedExpr = new PlusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void ParsePrefixExpression_SingleMinusOperator_ConstantOperands() {
        // Arrange
        var prefixFormula = "- 1 2";
        var expectedExpr = new MinusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void ParsePrefixExpression_SingleMultiplyOperator_ConstantOperands() {
        // Arrange
        var prefixFormula = "* 1 2";
        var expectedExpr = new MultiplyExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void ParsePrefixExpression_SingleDivideOperator_ConstantOperands() {
        // Arrange
        var prefixFormula = "/ 1 2";
        var expectedExpr = new DivideExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    // Parsing all unary operators

    [Fact]
    [Trait("Category", "Basic-UnaryOperators")]
    public void ParsePrefixExpression_SingleUnaryMinusOperator_ConstantOperand() {
        // Arrange
        var prefixFormula = "~ 1";
        var expectedExpr = new UnaryMinusExpression {
            Operand = new ConstantExpression(1),
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    // Detecting invalid inputs

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_EmptyFormula_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_SpacesOnly_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "    ";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_MissingOperator_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "1 2";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_MissingInnerOperand_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "+ 1 + 2";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_MissingOuterOperand_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "+ + 1 2";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_ExcessiveOperator_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "+ 1 2 +";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_ExcessiveOperand_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "+ 1 2 3";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_MissingRootOperator_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "+ 1 2 + 3 4";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_InvalidNumberPrefixInFormula_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "+ abc123 2";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_InvalidNumberSufixInFormula_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "+ 123abc 2";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_NotNumberValueInFormula_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "+ abc 2";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    [Fact]
    [Trait("Category", "InvalidInput")]
    public void ParsePrefixExpression_JustNotNumberValue_ShouldReturnNull() {
        // Arrange
        var prefixFormula = "abc";

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Null(expr);
    }

    // Resilient to additional spaces

    [Fact]
    [Trait("Category", "AdditionalSpaces")]
    public void ParsePrefixExpression_SingleOperator_LeadingSpaces() {
        // Arrange
        var prefixFormula = "   + 1 2";
        var expectedExpr = new PlusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "AdditionalSpaces")]
    public void ParsePrefixExpression_SingleOperator_TrailingSpaces() {
        // Arrange
        var prefixFormula = "+ 1 2   ";
        var expectedExpr = new PlusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "AdditionalSpaces")]
    public void ParsePrefixExpression_SingleOperator_InnerSpaces() {
        // Arrange
        var prefixFormula = "+  1   2";
        var expectedExpr = new PlusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "AdditionalSpaces")]
    public void ParsePrefixExpression_SingleOperator_AllLeadingTrailingInnerSpaces() {
        // Arrange
        var prefixFormula = " +  1   2 ";
        var expectedExpr = new PlusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    // Complex expression

    [Fact]
    [Trait("Category", "Complex")]
    public void ParsePrefixExpression_LeftChain_EvenOperatorDepth_ComplexExpression() {
        // Arrange
        var prefixFormula = "+ - + - 1 2 3 4 5";
        var expectedExpr = new PlusExpression {
            LeftOperand = new MinusExpression {
                LeftOperand = new PlusExpression {
                    LeftOperand = new MinusExpression {
                        LeftOperand = new ConstantExpression(1),
                        RightOperand = new ConstantExpression(2)
                    },
                    RightOperand = new ConstantExpression(3)
                },
                RightOperand = new ConstantExpression(4)
            },
            RightOperand = new ConstantExpression(5)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Complex")]
    public void ParsePrefixExpression_LeftChain_OddOperatorDepth_ComplexExpression() {
        // Arrange
        var prefixFormula = "- + - 1 2 3 4";
        var expectedExpr = new MinusExpression {
            LeftOperand = new PlusExpression {
                LeftOperand = new MinusExpression {
                    LeftOperand = new ConstantExpression(1),
                    RightOperand = new ConstantExpression(2)
                },
                RightOperand = new ConstantExpression(3)
            },
            RightOperand = new ConstantExpression(4)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Complex")]
    public void ParsePrefixExpression_RightChain_EvenOperatorDepth_ComplexExpression() {
        // Arrange
        var prefixFormula = "+ 1 - 2 + 3 - 4 5";
        var expectedExpr = new PlusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new MinusExpression {
                LeftOperand = new ConstantExpression(2),
                RightOperand = new PlusExpression {
                    LeftOperand = new ConstantExpression(3),
                    RightOperand = new MinusExpression {
                        LeftOperand = new ConstantExpression(4),
                        RightOperand = new ConstantExpression(5)
                    }
                }
            }
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Complex")]
    public void ParsePrefixExpression_RightChain_OddOperatorDepth_ComplexExpression() {
        // Arrange
        var prefixFormula = "- 1 + 2 - 3 4";
        var expectedExpr = new MinusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new PlusExpression {
                LeftOperand = new ConstantExpression(2),
                RightOperand = new MinusExpression {
                    LeftOperand = new ConstantExpression(3),
                    RightOperand = new ConstantExpression(4)
                }
            }
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    // Reference inputs

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void ParsePrefixExpression_ReferenceInput1_ComplexExpression() {
        // Arrange
        var prefixFormula = "+ ~ 1 3";
        var expectedExpr = new PlusExpression {
            LeftOperand = new UnaryMinusExpression {
                Operand = new ConstantExpression(1)
            },
            RightOperand = new ConstantExpression(3)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void ParsePrefixExpression_ReferenceInput2_ComplexExpression() {
        // Arrange
        var prefixFormula = "/ + - 5 2 * 2 + 3 3 ~ 2";
        var expectedExpr = new DivideExpression {
            LeftOperand = new PlusExpression {
                LeftOperand = new MinusExpression {
                    LeftOperand = new ConstantExpression(5),
                    RightOperand = new ConstantExpression(2)
                },
                RightOperand = new MultiplyExpression {
                    LeftOperand = new ConstantExpression(2),
                    RightOperand = new PlusExpression {
                        LeftOperand = new ConstantExpression(3),
                        RightOperand = new ConstantExpression(3)
                    }
                }
            },
            RightOperand = new UnaryMinusExpression {
                Operand = new ConstantExpression(2)
            }
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void ParsePrefixExpression_ReferenceInput3_ComplexExpression() {
        // Arrange
        const int bigValue1 = 2_000_000_000;
        const int bigValue2 = 2_100_000_000;

        var prefixFormula = $"- - {bigValue1} {bigValue2} {bigValue2}";
        var expectedExpr = new MinusExpression {
            LeftOperand = new MinusExpression {
                LeftOperand = new ConstantExpression(bigValue1),
                RightOperand = new ConstantExpression(bigValue2)
            },
            RightOperand = new ConstantExpression(bigValue2)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void ParsePrefixExpression_ReferenceInput4_ComplexExpression() {
        // Arrange
        var prefixFormula = "/ 100 - + 10 10 20";
        var expectedExpr = new DivideExpression {
            LeftOperand = new ConstantExpression(100),
            RightOperand = new MinusExpression {
                LeftOperand = new PlusExpression {
                    LeftOperand = new ConstantExpression(10),
                    RightOperand = new ConstantExpression(10)
                },
                RightOperand = new ConstantExpression(20)
            }
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void ParsePrefixExpression_MultidigitConstants_ModifiedReferenceInput1_ComplexExpression() {
        // Arrange
        var prefixFormula = "+ ~ 11 322";
        var expectedExpr = new PlusExpression {
            LeftOperand = new UnaryMinusExpression {
                Operand = new ConstantExpression(11)
            },
            RightOperand = new ConstantExpression(322)
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void ParsePrefixExpression_MultidigitConstants_ModifiedReferenceInput2_ComplexExpression() {
        // Arrange
        var prefixFormula = "/ + - 51 222 * 2333 + 34444 355555 ~ 2666666";
        var expectedExpr = new DivideExpression {
            LeftOperand = new PlusExpression {
                LeftOperand = new MinusExpression {
                    LeftOperand = new ConstantExpression(51),
                    RightOperand = new ConstantExpression(222)
                },
                RightOperand = new MultiplyExpression {
                    LeftOperand = new ConstantExpression(2333),
                    RightOperand = new PlusExpression {
                        LeftOperand = new ConstantExpression(34444),
                        RightOperand = new ConstantExpression(355555)
                    }
                }
            },
            RightOperand = new UnaryMinusExpression {
                Operand = new ConstantExpression(2666666)
            }
        };

        // Act
        var expr = ExpressionParser.ParsePrefixExpression(prefixFormula);

        // Assert
        Assert.Equivalent(expectedExpr, expr);
    }
}