﻿using ExpressionEvaluatorI_2024;
using Xunit;

namespace ExpressionEvaluatorI_2024_UnitTests;

public class ExpressionTests {
    // Basic values

    [Fact]
    [Trait("Category", "Basic-Values")]
    public void Evaluate_ConstantValue() {
        // Arrange
        var expr = new ConstantExpression(123);
        var expectedResult = 123;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    // Binary operators - non-overflowing evaluation

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void Evaluate_SinglePlusOperator_ConstantOperands() {
        // Arrange
        var expr = new PlusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };
        var expectedResult = 3;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void Evaluate_SingleMinusOperator_ConstantOperands() {
        // Arrange
        var expr = new MinusExpression {
            LeftOperand = new ConstantExpression(1),
            RightOperand = new ConstantExpression(2)
        };
        var expectedResult = -1;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void Evaluate_SingleMinusOperator_MinIntegerShouldNotOverflow() {
        // Arrange
        var expr = new MinusExpression {
            LeftOperand = new UnaryMinusExpression {
                Operand = new ConstantExpression(1)
            },
            RightOperand = new ConstantExpression(int.MaxValue)
        };
        var expectedResult = int.MinValue;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void Evaluate_SingleMultiplyOperator_ConstantOperands() {
        // Arrange
        var expr = new MultiplyExpression {
            LeftOperand = new ConstantExpression(2),
            RightOperand = new ConstantExpression(3)
        };
        var expectedResult = 6;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void Evaluate_SingleDivideOperator_IntegerResult_ConstantOperands() {
        // Arrange
        var expr = new DivideExpression {
            LeftOperand = new ConstantExpression(6),
            RightOperand = new ConstantExpression(2)
        };
        var expectedResult = 3;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void Evaluate_SingleDivideOperator_TruncatedRealResult_ConstantOperands() {
        // Arrange
        var expr = new DivideExpression {
            LeftOperand = new ConstantExpression(3),
            RightOperand = new ConstantExpression(2)
        };
        var expectedResult = 1;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void Evaluate_SingleDivideOperator_ZeroDividedByNonzero() {
        // Arrange
        var expr = new DivideExpression {
            LeftOperand = new ConstantExpression(0),
            RightOperand = new ConstantExpression(3)
        };
        var expectedResult = 0;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators")]
    public void Evaluate_SingleDivideOperator_DivisionByZero_ShouldThrowDivideByZeroException() {
        // Arrange
        var expr = new DivideExpression {
            LeftOperand = new ConstantExpression(3),
            RightOperand = new ConstantExpression(0)
        };

        // Act + Assert
        Assert.Throws<DivideByZeroException>(() => expr.Evaluate());
    }

    // Binary operators - evaluation with arithmetic overflow

    [Fact]
    [Trait("Category", "Basic-BinaryOperators-ArithmeticOverflow")]
    public void Evaluate_SinglePlusOperator_ArithmeticOverflow() {
        // Arrange
        var expr = new PlusExpression {
            LeftOperand = new ConstantExpression(int.MaxValue),
            RightOperand = new ConstantExpression(1)
        };
			
        // Act + Assert
        Assert.Throws<OverflowException>(() => expr.Evaluate());
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators-ArithmeticOverflow")]
    public void Evaluate_SingleMinusOperator_ArithmeticOverflow() {
        // Arrange
        var expr = new MinusExpression {
            LeftOperand = new UnaryMinusExpression {
                Operand = new ConstantExpression(2)
            },
            RightOperand = new ConstantExpression(int.MaxValue)
        };

        // Act + Assert
        Assert.Throws<OverflowException>(() => expr.Evaluate());
    }

    [Fact]
    [Trait("Category", "Basic-BinaryOperators-ArithmeticOverflow")]
    public void Evaluate_SingleMultiplyOperator_ArithmeticOverflow() {
        // Arrange
        var expr = new MultiplyExpression {
            LeftOperand = new ConstantExpression(2),
            RightOperand = new ConstantExpression(int.MaxValue)
        };

        // Act + Assert
        Assert.Throws<OverflowException>(() => expr.Evaluate());
    }

    // Unary operators - non-overflowing evaluation

    [Fact]
    [Trait("Category", "Basic-UnaryOperators")]
    public void Evaluate_SingleUnaryMinusOperator_ConstantOperand() {
        // Arrange
        var expr = new UnaryMinusExpression {
            Operand = new ConstantExpression(2)
        };
        var expectedResult = -2;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Basic-UnaryOperators")]
    public void Evaluate_SingleUnaryMinusOperator_NegatingMaxIntegerShouldNotOverflow() {
        // Arrange
        var expr = new UnaryMinusExpression {
            Operand = new ConstantExpression(int.MaxValue)
        };
        var expectedResult = -int.MaxValue;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    // Unary operators - evaluation with arithmetic overflow

    [Fact]
    [Trait("Category", "Basic-UnaryOperators-ArithmeticOverflow")]
    public void Evaluate_UnaryMinusOperator_ArithmeticOverflow() {
        // Arrange
        var expr = new UnaryMinusExpression {
            Operand = new MinusExpression {
                LeftOperand = new UnaryMinusExpression {
                    Operand = new ConstantExpression(1)
                },
                RightOperand = new ConstantExpression(int.MaxValue)
            }
        };

        // NOTE: We should not try to create "new UnaryMinusExpression { Operand = new ConstantExpression(int.MinValue) }"
        //	     instead of the expression tree above, as valid constants should be non-negative!

        // Act + Assert
        Assert.Throws<OverflowException>(() => expr.Evaluate());
    }

    // Reference inputs

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void Evaluate_ReferenceInput1_ComplexExpression() {
        // Arrange
        var expr = new PlusExpression {
            LeftOperand = new UnaryMinusExpression {
                Operand = new ConstantExpression(1)
            },
            RightOperand = new ConstantExpression(3)
        };
        var expectedResult = 2;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void Evaluate_ReferenceInput2_ComplexExpression() {
        // Arrange
        var expr = new DivideExpression {
            LeftOperand = new PlusExpression {
                LeftOperand = new MinusExpression {
                    LeftOperand = new ConstantExpression(5),
                    RightOperand = new ConstantExpression(2)
                },
                RightOperand = new MultiplyExpression {
                    LeftOperand = new ConstantExpression(2),
                    RightOperand = new PlusExpression {
                        LeftOperand = new ConstantExpression(3),
                        RightOperand = new ConstantExpression(3)
                    }
                }
            },
            RightOperand = new UnaryMinusExpression {
                Operand = new ConstantExpression(2)
            }
        };
        var expectedResult = -7;

        // Act
        var result = expr.Evaluate();

        // Assert
        Assert.Equal(expectedResult, result);
    }

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void Evaluate_ReferenceInput3_ComplexExpression() {
        // Arrange
        const int bigValue1 = 2_000_000_000;
        const int bigValue2 = 2_100_000_000;

        var expr = new MinusExpression {
            LeftOperand = new MinusExpression {
                LeftOperand = new ConstantExpression(bigValue1),
                RightOperand = new ConstantExpression(bigValue2)
            },
            RightOperand = new ConstantExpression(bigValue2)
        };

        // Act + Assert
        Assert.Throws<OverflowException>(() => expr.Evaluate());
    }

    [Fact]
    [Trait("Category", "Complex-ReferenceInput")]
    public void Evaluate_ReferenceInput4_ComplexExpression() {
        // Arrange
        var expr = new DivideExpression {
            LeftOperand = new ConstantExpression(100),
            RightOperand = new MinusExpression {
                LeftOperand = new PlusExpression {
                    LeftOperand = new ConstantExpression(10),
                    RightOperand = new ConstantExpression(10)
                },
                RightOperand = new ConstantExpression(20)
            }
        };

        // Act + Assert
        Assert.Throws<DivideByZeroException>(() => expr.Evaluate());
    }
}