﻿namespace ExpressionEvaluatorI_2024;

public abstract class Expression {
    public abstract int Evaluate();
}

// Value leaf node infrastructure:

public abstract class ValueExpression : Expression {
    public abstract int Value { get; }

    public sealed override int Evaluate() => Value;
}

// Value node types:

public sealed class ConstantExpression(int value) : ValueExpression {
    public override int Value { get; } = value;
}

public sealed class AlternativeConstantExpression(int _value) : ValueExpression {
    public override int Value => _value; // Captured primary constructor parameter!
}

// Operator infrastructure:

public abstract class OperatorExpression : Expression {
    public abstract bool AddOperand(Expression operand);
}

public abstract class UnaryExpression : OperatorExpression
{
    private Expression? _operand;
    public Expression Operand
    {
        get => _operand!;
        init => _operand = value;
    }

    public sealed override bool AddOperand(Expression operand) {
        _operand ??= operand;
        return true;
    }

    public sealed override int Evaluate() {
        return EvaluateUnaryOperator(Operand.Evaluate());
    }

    protected abstract int EvaluateUnaryOperator(int value);
}

public abstract class BinaryExpression : OperatorExpression
{
    private Expression? _leftOperand;
    public Expression LeftOperand
    {
        get => _leftOperand!;
        init => _leftOperand = value;
    }

    private Expression? _rightOperand;
    public Expression RightOperand
    {
        get => _rightOperand!;
        init => _rightOperand = value;
    }
    
    public sealed override bool AddOperand(Expression operand) {
        if (_leftOperand == null) {
            _leftOperand = operand;
            return false;
        } else {
            _rightOperand ??= operand;
            return true;
        }
    }

    public sealed override int Evaluate() {
        return EvaluateBinaryOperator(LeftOperand.Evaluate(), RightOperand.Evaluate());
    }

    protected abstract int EvaluateBinaryOperator(int leftValue, int rightValue);
}

// Binary operators:

public sealed class PlusExpression : BinaryExpression {
    protected override int EvaluateBinaryOperator(int leftValue, int rightValue) {
        return checked(leftValue + rightValue);
    }
}

public sealed class MinusExpression : BinaryExpression {
    protected override int EvaluateBinaryOperator(int leftValue, int rightValue) {
        return checked(leftValue - rightValue);
    }
}

public sealed class MultiplyExpression : BinaryExpression {
    protected override int EvaluateBinaryOperator(int leftValue, int rightValue) {
        return checked(leftValue * rightValue);
    }
}

public sealed class DivideExpression : BinaryExpression {
    protected override int EvaluateBinaryOperator(int leftValue, int rightValue) {
        return checked(leftValue / rightValue);    // Can generate DivideByZeroException
    }
}

// Unary operators:

public sealed class UnaryMinusExpression : UnaryExpression {
    protected override int EvaluateUnaryOperator(int value) {
        return checked(-value);
    }
}

// Prefix notation parser:

public static class ExpressionParser {
    private static readonly char[] s_WhitespaceSeparators = [' '];

    private static Expression? CreateExpressionFromToken(string token) => token switch {
        // Binary operators:
        "+" => new PlusExpression(),
        "-" => new MinusExpression(),
        "*" => new MultiplyExpression(),
        "/" => new DivideExpression(),

        // Unary operators:
        "~" => new UnaryMinusExpression(),

        // Values:
        _ when int.TryParse(token, out var value) => value >= 0
            ? new ConstantExpression(value)
            : null,	// Negative values are not valid input

        // Invalid token format:
        _ => null
    };

    public static Expression? ParsePrefixExpression(string exprString) {
        var tokens = exprString.Split(s_WhitespaceSeparators, StringSplitOptions.RemoveEmptyEntries);

        Expression? result = null;
        var unresolved = new Stack<OperatorExpression>();
        foreach (var token in tokens) {
            if (result != null) {
                // We correctly parsed the whole tree, but there was at least one more unprocessed token left.
                // This implies incorrect input, thus return null.

                return null;
            }

            switch (CreateExpressionFromToken(token)) {
                case null:
                    return null;    // Invalid token format = incorrect input, thus return null.

                case OperatorExpression operatorExpr:
                    unresolved.Push(operatorExpr);
                    break;

                case ValueExpression valueExpr:

                    Expression expr = valueExpr;
                    while (unresolved.Count > 0) {
                        OperatorExpression operatorExpr = unresolved.Peek();
                        if (operatorExpr.AddOperand(expr)) {
                            unresolved.Pop();
                            expr = operatorExpr;
                        } else {
                            break;
                        }
                    }

                    if (unresolved.Count == 0) {
                        result = expr;	// Candidate for result, just need to check no more tokens left
                    }

                    break;

                default:
                    throw new InvalidOperationException($"Unexpected expression type instance returned from {nameof(CreateExpressionFromToken)}() method.");
            }
        }

        return result;
    }
}

// Assignment solution:
internal class Program {
    private static void Main() {
        var inputLine = Console.ReadLine()!;	// Assignment: at least one line is always present on standard input, we do not need to handle empty input

        var expr = ExpressionParser.ParsePrefixExpression(inputLine);

        if (expr == null) {
            Console.WriteLine("Format Error");
            return;
        }

        try {
            Console.WriteLine(expr.Evaluate().ToString());
        } catch (DivideByZeroException) {
            Console.WriteLine("Divide Error");
        } catch (OverflowException) {
            Console.WriteLine("Overflow Error");
        }
    }
}
