﻿namespace ExpressionEvaluatorI_DirectEvaluationViaPatternMatching;

public abstract class Expression;

// Value leaf node infrastructure:

public abstract class ValueExpression : Expression {
    public abstract int Value { get; }
}

// Value node types:

public sealed class ConstantExpression(int _value) : ValueExpression
{
    public override int Value => _value;
}

// Operator infrastructure:

public abstract class OperatorExpression : Expression {
    public abstract bool AddOperand(Expression operand);
}

public abstract class UnaryExpression : OperatorExpression {
    private Expression? _operand;

    public Expression Operand
    {
        get => _operand!;
        init => _operand = value;
    }

    public sealed override bool AddOperand(Expression operand) {
        _operand ??= operand;
        return true;
    }
}

public abstract class BinaryExpression : OperatorExpression {
    private Expression? _leftOperand, _rightOperand;

    public Expression LeftOperand
    {
        get => _leftOperand!;
        init => _leftOperand = value;
    }
    public Expression RightOperand
    {
        get => _rightOperand!;
        init => _rightOperand = value;
    }

    public sealed override bool AddOperand(Expression operand) {
        if (_leftOperand == null) {
            _leftOperand = operand;
            return false;
        } else {
            _rightOperand ??= operand;
            return true;
        }
    }
}

// Binary operators:

public sealed class PlusExpression : BinaryExpression;

public sealed class MinusExpression : BinaryExpression;

public sealed class MultiplyExpression : BinaryExpression;

public sealed class DivideExpression : BinaryExpression;

// Unary operators:

public sealed class UnaryMinusExpression : UnaryExpression;

// Prefix notation parser:

public static class ExpressionParser {
    private static readonly char[] s_WhitespaceSeparators = [' '];

    private static Expression? CreateExpressionFromToken(string token) => token switch {
        // Binary operators:
        "+" => new PlusExpression(),
        "-" => new MinusExpression(),
        "*" => new MultiplyExpression(),
        "/" => new DivideExpression(),

        // Unary operators:
        "~" => new UnaryMinusExpression(),

        // Values:
        _ when int.TryParse(token, out int value) =>
            value >= 0 ?
                new ConstantExpression(value)
                :
                null,   // Negative values are not valid input

        // Invalid token format:
        _ => null
    };

    public static Expression? ParsePrefixExpression(string exprString) {
        string[] tokens = exprString.Split(s_WhitespaceSeparators, StringSplitOptions.RemoveEmptyEntries);

        Expression? result = null;
        var unresolved = new Stack<OperatorExpression>();
        foreach (string token in tokens) {
            if (result != null) {
                // We correctly parsed the whole tree, but there was at least one more unprocessed token left.
                // This implies incorrect input, thus return null.

                return null;
            }

            switch (CreateExpressionFromToken(token)) {
                case null:
                    return null;    // Invalid token format = incorrect input, thus return null.

                case OperatorExpression operatorExpr:
                    unresolved.Push(operatorExpr);
                    break;

                case ValueExpression valueExpr:

                    Expression expr = valueExpr;
                    while (unresolved.Count > 0) {
                        OperatorExpression operatorExpr = unresolved.Peek();
                        if (operatorExpr.AddOperand(expr)) {
                            unresolved.Pop();
                            expr = operatorExpr;
                        } else {
                            break;
                        }
                    }

                    if (unresolved.Count == 0) {
                        result = expr;  // Candidate for result, just need to check no more tokens left
                    }

                    break;

                default:
                    throw new InvalidOperationException($"Unexpected expression type instance returned from {nameof(CreateExpressionFromToken)}() method.");
            }
        }

        return result;
    }
}

// Assignment solution:

class Program {
    static int Evaluate(Expression expr) => expr switch {
        ValueExpression valueExpr => valueExpr.Value,

        // Binary expressions:
        BinaryExpression binaryExpr => binaryExpr switch {
            PlusExpression => checked(Evaluate(binaryExpr.LeftOperand) + Evaluate(binaryExpr.RightOperand)),
            MinusExpression => checked(Evaluate(binaryExpr.LeftOperand) - Evaluate(binaryExpr.RightOperand)),
            MultiplyExpression => checked(Evaluate(binaryExpr.LeftOperand) * Evaluate(binaryExpr.RightOperand)),
            DivideExpression => checked(Evaluate(binaryExpr.LeftOperand) / Evaluate(binaryExpr.RightOperand)),  // Can generate DivideByZeroException

            // If not present -> Warning: The switch expression does not handle all possible values of its input type(it is not exhaustive). For example, the pattern '_' is not covered.
            _ => throw new NotSupportedException($"Binary operator {binaryExpr.GetType().Name} is not supported.")	
        },
			
        // Unary expressions:
        UnaryExpression unaryExpr => unaryExpr switch {
            UnaryMinusExpression => checked(-Evaluate(unaryExpr.Operand)),
            _ => throw new NotSupportedException($"Unary operator {unaryExpr.GetType().Name} is not supported.")
        },

        _ => throw new NotSupportedException($"Expression type {expr.GetType().Name} is not supported.")
    };

    static void Main(string[] args) {
        string inputLine = Console.ReadLine()!; // Assignment: at least one line is always present on standard input, we do not need to handle empty input

        Expression? expr = ExpressionParser.ParsePrefixExpression(inputLine);

        if (expr == null) {
            Console.WriteLine("Format Error");
            return;
        }

        try {
            Console.WriteLine(Evaluate(expr).ToString());
        } catch (DivideByZeroException) {
            Console.WriteLine("Divide Error");
        } catch (OverflowException) {
            Console.WriteLine("Overflow Error");
        }
    }
}